#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "libusb.h"

#include <QtCore>
#include <QtGui>
#include <QtDebug>

//
// Some headers are system-specific
//
#ifdef WIN
#include <dbt.h>
#else
#include "libudev.h"
#endif

MainWindow::MainWindow(QWidget *parent) : QMainWindow(parent), ui(new Ui::MainWindow)
{
    qDebug() << "Start";

    ui->setupUi(this);
    this->setWindowFlags( Qt::Window | Qt::CustomizeWindowHint | Qt::WindowTitleHint | Qt::WindowCloseButtonHint );

    connect(this, &MainWindow::signal_DeviceConnected,      this,  &MainWindow::DeviceConnected );
    connect(this, &MainWindow::signal_DeviceDisconnected,   this,  &MainWindow::DeviceDisconnected );

    connect(this->ui->slider_R, &QSlider::valueChanged,  this,  &MainWindow::slider_Change );
    connect(this->ui->slider_G, &QSlider::valueChanged,  this,  &MainWindow::slider_Change );
    connect(this->ui->slider_B, &QSlider::valueChanged,  this,  &MainWindow::slider_Change );

    // Create timer for periodic hardware state check
    this->queryTimer = new QTimer(this);
    connect(this->queryTimer, &QTimer::timeout, this, &MainWindow::queryTimerTick );

#ifdef WIN
    //
    // Windows specific code - register for windows messages: device add/remove notification

    // Register for device connect notification
    DEV_BROADCAST_DEVICEINTERFACE devInt;
    ZeroMemory( &devInt, sizeof(devInt) );
    devInt.dbcc_size        = sizeof(DEV_BROADCAST_DEVICEINTERFACE);
    devInt.dbcc_devicetype  = DBT_DEVTYP_DEVICEINTERFACE;
    devInt.dbcc_classguid   = GUID_DEVINTERFACE_LAUNCHPAD;
    //
    HDEVNOTIFY m_hDeviceNotify = RegisterDeviceNotification((HANDLE)winId(),&devInt, DEVICE_NOTIFY_WINDOW_HANDLE );
    //
    if(m_hDeviceNotify == NULL)
    {
        qDebug() << "Error: Failed to register device notification!";
        qApp->quit();
    }
#else
    //
    // Linux specific code - start udev monitor

    // Create the udev object
    if (!(udev = udev_new()))
    {
        qDebug() << "Error: Failed to create udev!";
        qApp->quit();
    }

    // Set up a monitor to monitor usb devices
    mon = udev_monitor_new_from_netlink(udev, "udev");
    // We want only to receive information about usb devices
    udev_monitor_filter_add_match_subsystem_devtype(mon, "usb", "usb_device");
    udev_monitor_enable_receiving(mon);
    fd = udev_monitor_get_fd(mon);

    // Start timer that will periodicaly check if hardware is connected
    monitorTimer = new QTimer(this);
    connect(monitorTimer, &QTimer::timeout, this, &MainWindow::monitorTimerTick );
    monitorTimer->start(250);
#endif

    // No device at the begining
    LaunchpadDevice = NULL;

    // Initialize libusb
    if (libusb_init(NULL) < 0)
    {
        qDebug() << "Failed to initialize libusb";
        qApp->quit();
    }

    //If we are debugging print all information
    // libusb_set_debug(NULL, LIBUSB_LOG_LEVEL_DEBUG);

    // Initialize interface components
    DeviceDisconnected();

    // Check if device is already connected to system
    DeviceConnected();
}

// Clean on quit
MainWindow::~MainWindow()
{
    DeviceDisconnected();
    libusb_exit(NULL);
#ifndef WIN
    monitorTimer->stop();
    udev_unref(udev);
#endif
    delete ui;
}

#ifdef WIN
// Function that receive messages
// This is windows-specific
bool MainWindow::nativeEvent(const QByteArray& eventType, void* message,
                             long* result)
{
    Q_UNUSED( result );
    Q_UNUSED( eventType );

    MSG* msg = reinterpret_cast<MSG*>(message);

    // Does this specific message interest us?
    if(msg->message == WM_DEVICECHANGE)
    {
        switch(msg->wParam)
        {
        case DBT_DEVICEARRIVAL:
            emit signal_DeviceConnected();
            break;

        case DBT_DEVICEREMOVECOMPLETE:
            emit signal_DeviceDisconnected();
            break;
        }
    }

    // Qt handles the rest
    return false;
}
#else
// Function pools messges from udev
// Linux specific
void MainWindow::monitorTimerTick()
{
    // Set up select
    fd_set fds;
    struct timeval tv;
    int ret;
    FD_ZERO(&fds);
    FD_SET(fd, &fds);
    tv.tv_sec = 0;
    tv.tv_usec = 0;

    ret = select(fd+1, &fds, NULL, NULL, &tv);

    // Check if our file descriptor has received data.
    if (ret > 0 && FD_ISSET(fd, &fds))
    {
        // We encounter some event!
        qDebug() << "[i] udev event";

        // Get the device
        if ( (dev = udev_monitor_receive_device(mon)) == NULL)
        {
            qDebug() << "[w] receive_device returned no device!";
            return;
        }
        // Now check if the device is our rgb_led launchpad

        const char* str_action = udev_device_get_action(dev);
        const char* str_PID = udev_device_get_property_value(dev, "ID_MODEL_ID");
        const char* str_VID = udev_device_get_property_value(dev, "ID_VENDOR_ID");

        //            struct udev_list_entry *list_entry;
        //            udev_list_entry_foreach(list_entry,udev_device_get_properties_list_entry(dev))
        //            {
        //                const char *name = udev_list_entry_get_name(list_entry);
        //                const char *value = udev_list_entry_get_value(list_entry);
        //                printf(" %s : %s \n",name,value);
        //                fflush(stdout);
        //            }

        if ( (str_PID == NULL) || (str_VID==NULL) || (str_action==NULL) )
        {
            qDebug() << "[e] udev null string!";
            exit(-1);
        }

        // Compare strings and send signals
        if ( (strcmp(STR_PRODUCT_ID,str_PID) == 0) && (strcmp(STR_VENDOR_ID,str_VID) == 0) )
        {
            if (strcmp("add",str_action) == 0)
            {
                emit signal_DeviceConnected();
            }
            else if (strcmp("remove",str_action) == 0)
            {
                emit signal_DeviceDisconnected();
            }
            else
                qDebug() << "[w] unknown device action!";
        }
        udev_device_unref(dev);
    }

    // If there are more events to process, do not wait for next tick!
    if (ret-1 > 0)
        monitorTimerTick();
}
#endif

// Event:
// Our device appeared in the system
void MainWindow::DeviceConnected()
{
    // List storing all detected devices
    libusb_device **devs;

    // Get all devices connected to system
    if (libusb_get_device_list(NULL, &devs) < 0)
    {
        qDebug() << "Failed to get devices list";
        return;
    }

    // Scan for LaunchpadDevice VID & PID
    libusb_device *dev;
    // retdev stores the first device detected
    libusb_device *retdev = NULL;
    int i = 0;
    int count = 0;
    while ( (dev = devs[i++]) != NULL )
    {
        struct libusb_device_descriptor desc;

        if (libusb_get_device_descriptor(dev, &desc) < 0)
        {
            qDebug() << "failed to get device descriptor";
            continue;
        }

        // Is this device the one we are looking for?
        if ( (desc.idVendor == VENDOR_ID) && (desc.idProduct == PRODUCT_ID) )
        {

            count++;
            if ( retdev == NULL )
                retdev = dev;
        }
    }

    if (count <= 0)
    {
        qDebug() << "Warning! No device found";
        return;
    }

    if (count > 1 )
    {
        qDebug() << "Warning! More than one device found. Using first from the list";
    }

    if (libusb_open(retdev, &this->LaunchpadDevice) < 0)
    {
        qDebug() << "Error! Failed to open device";
        return;
    }

    libusb_free_device_list(devs, 1);

    // now connect to interface

    if (libusb_claim_interface(this->LaunchpadDevice, 0) != LIBUSB_SUCCESS)
    {
        qDebug() << "Error! Failed to claim interface";
        return;
    }

    qDebug() << "DeviceConnected";

    // If all ok enable control
    this->ui->slider_R->setEnabled(true);
    this->ui->slider_G->setEnabled(true);
    this->ui->slider_B->setEnabled(true);
    this->ui->button_ONOFF->setEnabled(true);

    // Send some data for embedded device initial conditions
    slider_Change();
    USBSendONOFF(false);

    // Start the timer to pool updates from device
    this->queryTimer->start(500);

    ui->statusBar->showMessage("Device connected.");
}

// Event:
// Device was disconnected
void MainWindow::DeviceDisconnected()
{
    qDebug() << "DeviceDisconnected";

    // Disable timer

    this->queryTimer->stop();

    // Disable controls and reset them to initial state

    this->ui->slider_R->setValue(200);
    this->ui->slider_G->setValue(200);
    this->ui->slider_B->setValue(200);
    this->ui->button_ONOFF->setChecked(false);

    this->ui->slider_R->setEnabled(false);
    this->ui->slider_G->setEnabled(false);
    this->ui->slider_B->setEnabled(false);
    this->ui->button_ONOFF->setEnabled(false);

    if(this->LaunchpadDevice != NULL)
    {
        libusb_release_interface(this->LaunchpadDevice, 0);
        libusb_close(this->LaunchpadDevice);
        this->LaunchpadDevice=NULL;
    }

    ui->statusBar->showMessage("Awaiting device connection.");
}

// Event:
// Update device when we change sliders values
void MainWindow::slider_Change()
{
    if (this->LaunchpadDevice!=NULL)
        USBSendRGB(this->ui->slider_R->value(),this->ui->slider_G->value(),this->ui->slider_B->value());
}

// Event:
// Toogle LED on button click
void MainWindow::on_button_ONOFF_clicked()
{
    // Enable/Disable LED according to button state
    USBSendONOFF(ui->button_ONOFF->isChecked());
}

// Periodically check if LED is still on on the device
void MainWindow::queryTimerTick()
{
    this->ui->button_ONOFF->setChecked(USBAskDevice());
}

// USB /////

void MainWindow::USBSendONOFF(bool on)
{
    uint8_t packet[1];
    int transferred = 0;

    if(this->LaunchpadDevice == NULL)
    {
        qDebug() << "Warning: Device is not availabe!";
        return;
    }

    // Should we enable or disable LEDs?
    if (on)
        packet[0] = 0x0E;
    else
        packet[0] = 0x0D;

    // Endpoint address can be seen in USBView.exe . 0x01 is 1st OUT endpoint in this case
    // We are sending packet. Transmitted bytes count should be same as
    // packet size and function should return no errors
    if ( (libusb_bulk_transfer(this->LaunchpadDevice, 0x01, packet, sizeof(packet) , &transferred, 0) != 0 )
         || (transferred != sizeof(packet)) )
    {
        qDebug() << "Error: Failed to send data";
        DeviceDisconnected();
        return;
    }
}

void MainWindow::USBSendRGB(uint16_t red, uint16_t green, uint16_t blue)
{
    uint8_t packet[7];
    int transferred = 0;

    if(this->LaunchpadDevice == NULL)
    {
        qDebug() << "Warning: Device is not availabe!";
        return;
    }

    // Construct packet
    packet[0] = 0x0C;
    packet[1] = (uint8_t)(red)      &0xFF;
    packet[2] = (uint8_t)(red>>8)   &0xFF;
    packet[3] = (uint8_t)(green)    &0xFF;
    packet[4] = (uint8_t)(green>>8) &0xFF;
    packet[5] = (uint8_t)(blue)     &0xFF;
    packet[6] = (uint8_t)(blue>>8)  &0xFF;

    // Endpoint address can be seen in USBView.exe . 0x01 is 1st OUT endpoint in this case
    // We are sending packet. Transmitted bytes count should be same as
    // packet size and function should return no errors
    if ( (libusb_bulk_transfer(this->LaunchpadDevice, 0x01, packet, sizeof(packet) , &transferred, 0) != 0 )
         || (transferred != sizeof(packet)) )
    {
        qDebug() << "Error: Failed to send data";
        DeviceDisconnected();
        return;
    }
}

// We need to periodically ask device for it's state as it cannot prompt when user pressed on/off button
// This function returns LEDs state
bool  MainWindow::USBAskDevice()
{
    uint8_t packet[1];
    int transferred = 0;

    if(this->LaunchpadDevice == NULL)
    {
        qDebug() << "Warning: Device is not availabe!";
        return false;
    }

    // create packet
    packet[0] = 0x0A;

    // Endpoint address can be seen in USBView.exe . 0x01 is 1st OUT endpoint in this case
    // We are sending packet. Transmitted bytes count should be same as
    // packet size and function should return no errors
    if ( (libusb_bulk_transfer(this->LaunchpadDevice, 0x01, packet, sizeof(packet) , &transferred, 0) != 0 )
         || (transferred != sizeof(packet)) )
    {
        qDebug() << "Error: Failed to send data";
        DeviceDisconnected();
        return false;
    }

    // Receive response ///

    // Endpoint address can be seen in USBView.exe . 0x81 is 1st IN endpoint in this case
    // We are receiving packet. Transmitted bytes count should be same as
    // packet size and function should return no errors
    if ( (libusb_bulk_transfer(this->LaunchpadDevice, 0x81, packet, sizeof(packet) , &transferred, 0) != 0 )
         || (transferred != sizeof(packet)) )
    {
        qDebug() << "Error: Failed to receive data";
        DeviceDisconnected();
        return false;
    }

    return ( packet[0] == 0 );
}
