//*****************************************************************************
//
// usb_dev_bulk.c - Main routines for the generic bulk device example.
//
// Copyright (c) 2012 Texas Instruments Incorporated.  All rights reserved.
// Software License Agreement
// 
// Texas Instruments (TI) is supplying this software for use solely and
// exclusively on TI's microcontroller products. The software is owned by
// TI and/or its suppliers, and is protected under applicable copyright
// laws. You may not combine this software with "viral" open-source
// software in order to form a larger program.
// 
// THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
// NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
// NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. TI SHALL NOT, UNDER ANY
// CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
// DAMAGES, FOR ANY REASON WHATSOEVER.
// 
// This is part of revision 9453 of the EK-LM4F120XL Firmware Package.
//
//*****************************************************************************

#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "driverlib/debug.h"
#include "driverlib/fpu.h"
#include "driverlib/gpio.h"
#include "driverlib/interrupt.h"
#include "driverlib/pin_map.h"
#include "driverlib/sysctl.h"
#include "driverlib/systick.h"
#include "driverlib/timer.h"
#include "driverlib/uart.h"
#include "driverlib/rom.h"
#include "usblib/usblib.h"
#include "usblib/usb-ids.h"
#include "usblib/device/usbdevice.h"
#include "usblib/device/usbdbulk.h"
#include "utils/uartstdio.h"
#include "utils/ustdlib.h"
#include "usb_bulk_structs.h"
#include "rgb.h"
#include "buttons.h"

//*****************************************************************************
//
//! \addtogroup example_list
//! <h1>USB Generic Bulk Device (usb_dev_bulk)</h1>
//!
//! This example provides a generic USB device offering simple bulk data
//! transfer to and from the host.  The device uses a vendor-specific class ID
//! and supports a single bulk IN endpoint and a single bulk OUT endpoint.
//! Data received from the host is assumed to be ASCII text and it is
//! echoed back with the case of all alphabetic characters swapped.
//!
//! A Windows INF file for the device is provided on the installation CD and
//! in the C:/StellarisWare/windows_drivers directory of StellarisWare
//! releases.  This INF contains information required to install the WinUSB
//! subsystem on WindowsXP and Vista PCs.  WinUSB is a Windows subsystem
//! allowing user mode applications to access the USB device without the need
//! for a vendor-specific kernel mode driver.
//!
//! A sample Windows command-line application, usb_bulk_example, illustrating
//! how to connect to and communicate with the bulk device is also provided.
//! The application binary is installed as part of the ``Windows-side examples
//! for USB kits'' package (SW-USB-win) on the installation CD or via download
//! from http://www.ti.com/stellarisware .  Project files are included to allow
//! the examples to be built using Microsoft VisualStudio 2008.  Source code
//! for this application can be found in directory
//! StellarisWare/tools/usb_bulk_example.
//
//*****************************************************************************

//*****************************************************************************
//
// The system tick rate expressed both as ticks per second and a millisecond
// period.
//
//*****************************************************************************
#define SYSTICKS_PER_SECOND     100
#define SYSTICK_PERIOD_MS       (1000 / SYSTICKS_PER_SECOND)

//*****************************************************************************
//
// The global system tick counter.
//
//*****************************************************************************
volatile unsigned long g_ulSysTickCount = 0;

//*****************************************************************************
//
// Variables tracking transmit and receive counts.
//
//*****************************************************************************
volatile unsigned long g_ulTxCount = 0;
volatile unsigned long g_ulRxCount = 0;
#ifdef DEBUG
unsigned long g_ulUARTRxErrors = 0;
#endif

//*****************************************************************************
//
// Debug-related definitions and declarations.
//
// Debug output is available via UART0 if DEBUG is defined during build.
//
//*****************************************************************************
#ifdef DEBUG
//*****************************************************************************
//
// Map all debug print calls to UARTprintf in debug builds.
//
//*****************************************************************************
#define DEBUG_PRINT UARTprintf

#else

//*****************************************************************************
//
// Compile out all debug print calls in release builds.
//
//*****************************************************************************
#define DEBUG_PRINT while(0) ((int (*)(char *, ...))0)
#endif

//*****************************************************************************
//
// Flags used to pass commands from interrupt context to the main loop.
//
//*****************************************************************************
#define COMMAND_PACKET_RECEIVED 0x00000001
#define COMMAND_STATUS_UPDATE   0x00000002

volatile unsigned long g_ulFlags = 0;
char *g_pcStatus;

//*****************************************************************************
//
// Global flag indicating that a USB configuration has been set.
//
//*****************************************************************************
static volatile tBoolean g_bUSBConfigured = false;

//*****************************************************************************
//
// Variable for button handling
//
//*****************************************************************************
volatile unsigned char button = 0;

//*****************************************************************************
//
// The error routine that is called if the driver library encounters an error.
//
//*****************************************************************************
#ifdef DEBUG
void
__error__(char *pcFilename, unsigned long ulLine)
{
	UARTprintf("Error at line %d of %s\n", ulLine, pcFilename);
	while(1)
	{
	}
}
#endif

//*****************************************************************************
//
// Interrupt handler for the system tick counter.
//
//*****************************************************************************
void SysTickIntHandler(void)
{
	//
	// Update our system tick counter.
	//
	g_ulSysTickCount++;

	//
	// Get button status
	// If button is down increase counter for debouncing
	//
	if ( ROM_GPIOPinRead(BUTTONS_GPIO_BASE, LEFT_BUTTON) == 0 )
		button++;
	else
		button=0;
}

//*****************************************************************************
//
// Handles bulk driver notifications related to the transmit channel (data to
// the USB host).
//
// \param pvCBData is the client-supplied callback pointer for this channel.
// \param ulEvent identifies the event we are being notified about.
// \param ulMsgValue is an event-specific value.
// \param pvMsgData is an event-specific pointer.
//
// This function is called by the bulk driver to notify us of any events
// related to operation of the transmit data channel (the IN channel carrying
// data to the USB host).
//
// \return The return value is event-specific.
//
//*****************************************************************************
unsigned long
TxHandler(void *pvCBData, unsigned long ulEvent, unsigned long ulMsgValue,
          void *pvMsgData)
{
    //
    // We are not required to do anything in response to any transmit event
    // in this example.
    //

    return(0);
}

//*****************************************************************************
//
// Receive new data and interpret it.
//
// \param psDevice points to the instance data for the device whose data is to
// be processed.
// \param pcData points to the newly received data in the USB receive buffer.
// \param ulNumBytes is the number of bytes of data available to be processed.
//
// This function is called whenever we receive a notification that data is
// available from the host. We read the data, byte-by-byte and swap the case
// of any alphabetical characters found then write it back out to be
// transmitted back to the host.
//
// \return Returns the number of bytes of data processed.
//
//*****************************************************************************
static unsigned long ProcessPacket(tUSBDBulkDevice *psDevice, unsigned char *pcData, unsigned long ulNumBytes)
{
	unsigned char value;

	// Read packet - first byte defines if there is more
	USBBufferRead(&g_sRxBuffer, (unsigned char*) &value, 1);

	switch (value)
	{
	case 0x0E:
		RGBEnable();
		break;

	case 0x0D:
		RGBDisable();
		break;

	case 0x0C:
		// Updating colour - read colour setting bytes
		USBBufferRead(&g_sRxBuffer, (unsigned char*) &g_ulColors, 6);
		RGBColorSet(g_ulColors);
		break;

	// Host ask for status
	case 0x0A:
		// Get status and send it
		value = RGBStatus();
		if ( USBBufferWrite(&g_sTxBuffer, &value,1 ) != 1 )
		{
			//
			// We are sending small amount of data so this
			// should never happen
			//
			UARTprintf("No space in USB TX buffer!\n");
		}
		break;

	default:
		UARTprintf("Unknown command from host! %04x\n", value);
	}
	USBBufferFlush(&g_sRxBuffer);

	//
	// We processed as much data as we can directly from the receive buffer so
	// we need to return the number of bytes to allow the lower layer to
	// update its read pointer appropriately.
	//
	return 0;
}

//*****************************************************************************
//
// Handles bulk driver notifications related to the receive channel (data from
// the USB host).
//
// \param pvCBData is the client-supplied callback pointer for this channel.
// \param ulEvent identifies the event we are being notified about.
// \param ulMsgValue is an event-specific value.
// \param pvMsgData is an event-specific pointer.
//
// This function is called by the bulk driver to notify us of any events
// related to operation of the receive data channel (the OUT channel carrying
// data from the USB host).
//
// \return The return value is event-specific.
//
//*****************************************************************************
unsigned long RxHandler(void *pvCBData, unsigned long ulEvent, unsigned long ulMsgValue, void *pvMsgData)
{
	//
	// Which event are we being sent?
	//
	switch (ulEvent)
	{
	//
	// We are connected to a host and communication is now possible.
	//
	case USB_EVENT_CONNECTED:
	{
		g_bUSBConfigured = true;
		UARTprintf("Host connected.\n");

		//
		// Flush our buffers.
		//
		USBBufferFlush(&g_sTxBuffer);
		USBBufferFlush(&g_sRxBuffer);

		break;
	}

		//
		// The host has disconnected.
		//
	case USB_EVENT_DISCONNECTED:
	{
		g_bUSBConfigured = false;
		UARTprintf("Host disconnected.\n");
		break;
	}

		//
		// A new packet has been received.
		//
	case USB_EVENT_RX_AVAILABLE:
	{
		//
		// Read the new packet and update RGB data.
		//
		return (ProcessPacket((tUSBDBulkDevice *) pvCBData, pvMsgData, ulMsgValue));
	}

		//
		// Ignore SUSPEND and RESUME for now.
		//
	case USB_EVENT_SUSPEND:
	case USB_EVENT_RESUME:
	{
		break;
	}

		//
		// Ignore all other events and return 0.
		//
	default:
	{
		break;
	}
	}

	return (0);
}

//*****************************************************************************
//
// This is the main application entry function.
//
//*****************************************************************************
int main(void)
{
	volatile unsigned long ulLoop;

	//
	// Enable lazy stacking for interrupt handlers.  This allows floating-point
	// instructions to be used within interrupt handlers, but at the expense of
	// extra stack usage.
	//
	ROM_FPULazyStackingEnable ();

	//
	// Set the clocking to run from the PLL at 50MHz
	//
	ROM_SysCtlClockSet (SYSCTL_SYSDIV_4 | SYSCTL_USE_PLL | SYSCTL_OSC_MAIN | SYSCTL_XTAL_16MHZ);

	//
	// Configure the relevant pins such that UART0 owns them.
	//
	ROM_SysCtlPeripheralEnable (SYSCTL_PERIPH_GPIOA);
	GPIOPinConfigure(GPIO_PA0_U0RX);
	GPIOPinConfigure(GPIO_PA1_U0TX);
	ROM_GPIOPinTypeUART (GPIO_PORTA_BASE, GPIO_PIN_0 | GPIO_PIN_1);

	//
	// Enable the GPIO port that is used for the on-board LED.
	//
	ROM_SysCtlPeripheralEnable (SYSCTL_PERIPH_GPIOF);

	//
	// Enable the GPIO pins for the LED.
	//
	RGBInit(0);
	RGBColorSet(g_ulColors);

	//
	// Set the buttons to be used
	//
	ButtonsInit();

	//
	// Open UART0 and show the application name on the UART.
	//
	UARTStdioInit(0);
	UARTprintf("\033[2JStellaris USB bulk device example\n");
	UARTprintf("---------------------------------\n\n");

	//
	// Not configured initially.
	//
	g_bUSBConfigured = false;

	//
	// Enable the GPIO peripheral used for USB, and configure the USB
	// pins.
	//
	ROM_SysCtlPeripheralEnable (SYSCTL_PERIPH_GPIOD);
	ROM_GPIOPinTypeUSBAnalog (GPIO_PORTD_BASE, GPIO_PIN_4 | GPIO_PIN_5);

	//
	// Enable the system tick.
	//
	ROM_SysTickPeriodSet (ROM_SysCtlClockGet () / SYSTICKS_PER_SECOND);
	ROM_SysTickIntEnable ();
	ROM_SysTickEnable ();

	//
	// Tell the user what we are up to.
	//
	UARTprintf("Configuring USB\n");

	//
	// Initialise the transmit and receive buffers.
	//
	USBBufferInit((tUSBBuffer *) &g_sTxBuffer);
	USBBufferInit((tUSBBuffer *) &g_sRxBuffer);

	//
	// Set the USB stack mode to Device mode with VBUS monitoring.
	//
	USBStackModeSet(0, USB_MODE_FORCE_DEVICE, 0);

	//
	// Pass our device information to the USB library and place the device
	// on the bus.
	//
	USBDBulkInit(0, (tUSBDBulkDevice *) &g_sBulkDevice);

	//
	// Wait for initial configuration to complete.
	//
	UARTprintf("Waiting for host...\n");

	//
	// Never ending loop
	//
	for (;;)
	{
		//
		// If the button down long enough (0.5 [s])
		// Switch LEDs off
		//

		if (button > 50)
		{
			RGBDisable();
		}
	}
}
